"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAppAnalyticsRouter = registerAppAnalyticsRouter;
var _configSchema = require("@osd/config-schema");
var _application_analytics = require("../../../common/constants/application_analytics");
var _app_analytics_adaptor = require("../../../server/adaptors/application_analytics/app_analytics_adaptor");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerAppAnalyticsRouter(router) {
  const appAnalyticsBackend = new _app_analytics_adaptor.AppAnalyticsAdaptor();

  // Fetches all existing applications
  router.get({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/`,
    validate: {}
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);
    let applicationsData = [];
    try {
      applicationsData = await appAnalyticsBackend.fetchApps(opensearchClient);
      return response.ok({
        body: {
          data: applicationsData
        }
      });
    } catch (err) {
      console.error('Error occurred while fetching applications', err);
      return response.custom({
        statusCode: err.statusCode || 400,
        body: err.message
      });
    }
  });

  // Fetch application by id
  router.get({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/{appId}`,
    validate: {
      params: _configSchema.schema.object({
        appId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const appObject = await appAnalyticsBackend.fetchAppById(opensearchClient, request.params.appId);
      return response.ok({
        body: appObject
      });
    } catch (err) {
      console.error('Error occurred while fetching application', err);
      return response.custom({
        statusCode: err.statusCode || 400,
        body: err.message
      });
    }
  });

  // Create a new application
  router.post({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        description: _configSchema.schema.maybe(_configSchema.schema.string()),
        baseQuery: _configSchema.schema.string(),
        servicesEntities: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        traceGroups: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        availabilityVisId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const newAppId = await appAnalyticsBackend.createNewApp(opensearchClient, request.body);
      return response.ok({
        body: {
          message: 'Application Created',
          newAppId
        }
      });
    } catch (err) {
      console.error('Error occurred while creating a new application', err);
      return response.custom({
        statusCode: err.statusCode || 400,
        body: err.message
      });
    }
  });

  // Renames an existing application
  router.put({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/rename`,
    validate: {
      body: _configSchema.schema.object({
        appId: _configSchema.schema.string(),
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      await appAnalyticsBackend.renameApp(opensearchClient, request.body.appId, request.body.name);
      return response.ok({
        body: {
          message: 'Application Renamed'
        }
      });
    } catch (err) {
      console.error('Error occurred while renaming an existing application', err);
      return response.custom({
        statusCode: err.statusCode || 400,
        body: err.message
      });
    }
  });

  // Updates an existing application
  router.put({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/`,
    validate: {
      body: _configSchema.schema.object({
        appId: _configSchema.schema.string(),
        updateBody: _configSchema.schema.object({
          name: _configSchema.schema.maybe(_configSchema.schema.string()),
          description: _configSchema.schema.maybe(_configSchema.schema.string()),
          servicesEntities: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
          traceGroups: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
          panelId: _configSchema.schema.maybe(_configSchema.schema.string()),
          availabilityVisId: _configSchema.schema.maybe(_configSchema.schema.string())
        })
      })
    }
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const updatedAppId = await appAnalyticsBackend.updateApp(opensearchClient, request.body.appId, request.body.updateBody);
      return response.ok({
        body: {
          message: 'Application Updated',
          updatedAppId
        }
      });
    } catch (err) {
      console.error('Error occurred while updating an existing application', err);
      return response.custom({
        statusCode: err.statusCode || 400,
        body: err.message
      });
    }
  });

  // Delete applications
  router.delete({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/{appList}`,
    validate: {
      params: _configSchema.schema.object({
        appList: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const delResponse = await appAnalyticsBackend.deleteApp(opensearchClient, request.params.appList);
      return response.ok({
        body: delResponse
      });
    } catch (err) {
      return response.custom({
        statusCode: err.statusCode || 400,
        body: err.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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