"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.clearParagraphs = clearParagraphs;
exports.createNotebook = createNotebook;
exports.createParagraphs = createParagraphs;
exports.deleteParagraphs = deleteParagraphs;
exports.fetchNotebook = fetchNotebook;
exports.runParagraph = runParagraph;
exports.updateFetchParagraph = updateFetchParagraph;
exports.updateParagraphs = updateParagraphs;
exports.updateRunFetchParagraph = updateRunFetchParagraph;
var _performanceNow = _interopRequireDefault(require("performance-now"));
var _uuid = require("uuid");
var _observability_saved_object_attributes = require("../../../common/types/observability_saved_object_attributes");
var _query_helpers = require("../../common/helpers/notebooks/query_helpers");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function createNotebook(paragraphInput, inputType) {
  try {
    let paragraphType = 'MARKDOWN';
    if (inputType === 'VISUALIZATION') {
      paragraphType = 'VISUALIZATION';
    }
    if (inputType === 'OBSERVABILITY_VISUALIZATION') {
      paragraphType = 'OBSERVABILITY_VISUALIZATION';
    }
    if (paragraphInput.substring(0, 3) === '%sql' || paragraphInput.substring(0, 3) === '%ppl') {
      paragraphType = 'QUERY';
    }
    const inputObject = {
      inputType: paragraphType,
      inputText: paragraphInput
    };
    const outputObjects = [{
      outputType: paragraphType,
      result: '',
      execution_time: '0s'
    }];
    const newParagraph = {
      id: 'paragraph_' + (0, _uuid.v4)(),
      dateCreated: new Date().toISOString(),
      dateModified: new Date().toISOString(),
      input: inputObject,
      output: outputObjects
    };
    return newParagraph;
  } catch (error) {
    throw new Error('Create Paragraph Error:' + error);
  }
}
async function fetchNotebook(noteId, opensearchNotebooksClient) {
  try {
    const notebook = await opensearchNotebooksClient.get(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, noteId);
    return notebook;
  } catch (error) {
    throw new Error('update Paragraph Error:' + error);
  }
}
async function createParagraphs(params, opensearchNotebooksClient) {
  const notebookinfo = await fetchNotebook(params.noteId, opensearchNotebooksClient);
  const paragraphs = notebookinfo.attributes.savedNotebook.paragraphs;
  const newParagraph = createNotebook(params.paragraphInput, params.inputType);
  paragraphs.splice(params.paragraphIndex, 0, newParagraph);
  const updateNotebook = {
    paragraphs,
    dateModified: new Date().toISOString()
  };
  await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, params.noteId, {
    savedNotebook: updateNotebook
  });
  await fetchNotebook(params.noteId, opensearchNotebooksClient);
  return newParagraph;
}
async function clearParagraphs(params, opensearchNotebooksClient) {
  const notebookinfo = await fetchNotebook(params.noteId, opensearchNotebooksClient);
  const updatedparagraphs = [];
  notebookinfo.attributes.savedNotebook.paragraphs.map(paragraph => {
    const updatedParagraph = {
      ...paragraph
    };
    updatedParagraph.output = [];
    updatedparagraphs.push(updatedParagraph);
  });
  const updateNotebook = {
    paragraphs: updatedparagraphs,
    dateModified: new Date().toISOString()
  };
  try {
    await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, params.noteId, {
      savedNotebook: updateNotebook
    });
    return {
      paragraphs: updatedparagraphs
    };
  } catch (error) {
    throw new Error('Clear Paragraph Error:' + error);
  }
}
async function deleteParagraphs(params, opensearchNotebooksClient) {
  const notebookinfo = await fetchNotebook(params.noteId, opensearchNotebooksClient);
  const updatedparagraphs = [];
  if (params.paragraphId !== undefined) {
    notebookinfo.attributes.savedNotebook.paragraphs.map(paragraph => {
      if (paragraph.id !== params.paragraphId) {
        updatedparagraphs.push(paragraph);
      }
    });
  }
  const updateNotebook = {
    paragraphs: updatedparagraphs,
    dateModified: new Date().toISOString()
  };
  try {
    await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, params.noteId, {
      savedNotebook: updateNotebook
    });
    return {
      paragraphs: updatedparagraphs
    };
  } catch (error) {
    throw new Error('update Paragraph Error:' + error);
  }
}
async function updateRunFetchParagraph(params, opensearchNotebooksClient) {
  try {
    const notebookinfo = await fetchNotebook(params.noteId, opensearchNotebooksClient);
    const updatedInputParagraphs = updateParagraphs(notebookinfo.attributes.savedNotebook.paragraphs, params.paragraphId, params.paragraphInput, params.paragraphType, params.dataSourceMDSId, params.dataSourceMDSLabel);
    const updatedOutputParagraphs = await runParagraph(updatedInputParagraphs, params.paragraphId);
    const updateNotebook = {
      paragraphs: updatedOutputParagraphs,
      dateModified: new Date().toISOString()
    };
    await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, params.noteId, {
      savedNotebook: updateNotebook
    });
    let resultParagraph = {};
    let index = 0;
    for (index = 0; index < updatedOutputParagraphs.length; ++index) {
      if (params.paragraphId === updatedOutputParagraphs[index].id) {
        resultParagraph = updatedOutputParagraphs[index];
      }
    }
    return resultParagraph;
  } catch (error) {
    throw new Error('Update/Run Paragraph Error:' + error);
  }
}
function runParagraph(paragraphs, paragraphId) {
  try {
    const updatedParagraphs = [];
    let index = 0;
    for (index = 0; index < paragraphs.length; ++index) {
      const startTime = (0, _performanceNow.default)();
      const updatedParagraph = {
        ...paragraphs[index]
      };
      if (paragraphs[index].id === paragraphId) {
        updatedParagraph.dateModified = new Date().toISOString();
        if ((0, _query_helpers.inputIsQuery)(paragraphs[index].input.inputText)) {
          updatedParagraph.output = [{
            outputType: 'QUERY',
            result: paragraphs[index].input.inputText.substring(4, paragraphs[index].input.inputText.length),
            execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
          }];
        } else if (paragraphs[index].input.inputText.substring(0, 3) === '%md') {
          updatedParagraph.output = [{
            outputType: 'MARKDOWN',
            result: paragraphs[index].input.inputText.substring(4, paragraphs[index].input.inputText.length),
            execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
          }];
        } else if (paragraphs[index].input.inputType === 'VISUALIZATION') {
          updatedParagraph.dateModified = new Date().toISOString();
          updatedParagraph.output = [{
            outputType: 'VISUALIZATION',
            result: '',
            execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
          }];
        } else if (paragraphs[index].input.inputType === 'OBSERVABILITY_VISUALIZATION') {
          updatedParagraph.dateModified = new Date().toISOString();
          updatedParagraph.output = [{
            outputType: 'OBSERVABILITY_VISUALIZATION',
            result: '',
            execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
          }];
        } else if ((0, _query_helpers.formatNotRecognized)(paragraphs[index].input.inputText)) {
          updatedParagraph.output = [{
            outputType: 'MARKDOWN',
            result: 'Please select an input type (%sql, %ppl, or %md)',
            execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
          }];
        }
      }
      updatedParagraphs.push(updatedParagraph);
    }
    return updatedParagraphs;
  } catch (error) {
    throw new Error('Running Paragraph Error:' + error);
  }
}
function updateParagraphs(paragraphs, paragraphId, paragraphInput, paragraphType, dataSourceMDSId, dataSourceMDSLabel) {
  try {
    const updatedParagraphs = [];
    paragraphs.map(paragraph => {
      const updatedParagraph = {
        ...paragraph
      };
      if (paragraph.id === paragraphId) {
        updatedParagraph.dataSourceMDSId = dataSourceMDSId;
        updatedParagraph.dataSourceMDSLabel = dataSourceMDSLabel;
        updatedParagraph.dateModified = new Date().toISOString();
        updatedParagraph.input.inputText = paragraphInput;
        if (paragraphType.length > 0) {
          updatedParagraph.input.inputType = paragraphType;
        }
      }
      updatedParagraphs.push(updatedParagraph);
    });
    return updatedParagraphs;
  } catch (error) {
    throw new Error('Update Paragraph Error:' + error);
  }
}
async function updateFetchParagraph(params, opensearchNotebooksClient) {
  try {
    const notebookinfo = await fetchNotebook(params.noteId, opensearchNotebooksClient);
    const updatedInputParagraphs = updateParagraphs(notebookinfo.attributes.savedNotebook.paragraphs, params.paragraphId, params.paragraphInput);
    const updateNotebook = {
      paragraphs: updatedInputParagraphs,
      dateModified: new Date().toISOString()
    };
    await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, params.noteId, {
      savedNotebook: updateNotebook
    });
    let resultParagraph = {};
    updatedInputParagraphs.map(paragraph => {
      if (params.paragraphId === paragraph.id) {
        resultParagraph = paragraph;
      }
    });
    return resultParagraph;
  } catch (error) {
    throw new Error('update Paragraph Error:' + error);
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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