﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/workspaces-web/WorkSpacesWebRequest.h>
#include <aws/workspaces-web/WorkSpacesWeb_EXPORTS.h>
#include <aws/workspaces-web/model/Tag.h>
#include <aws/workspaces-web/model/WebContentFilteringPolicy.h>

#include <utility>

namespace Aws {
namespace WorkSpacesWeb {
namespace Model {

/**
 */
class CreateBrowserSettingsRequest : public WorkSpacesWebRequest {
 public:
  AWS_WORKSPACESWEB_API CreateBrowserSettingsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateBrowserSettings"; }

  AWS_WORKSPACESWEB_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The tags to add to the browser settings resource. A tag is a key-value
   * pair.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateBrowserSettingsRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateBrowserSettingsRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The custom managed key of the browser settings.</p>
   */
  inline const Aws::String& GetCustomerManagedKey() const { return m_customerManagedKey; }
  inline bool CustomerManagedKeyHasBeenSet() const { return m_customerManagedKeyHasBeenSet; }
  template <typename CustomerManagedKeyT = Aws::String>
  void SetCustomerManagedKey(CustomerManagedKeyT&& value) {
    m_customerManagedKeyHasBeenSet = true;
    m_customerManagedKey = std::forward<CustomerManagedKeyT>(value);
  }
  template <typename CustomerManagedKeyT = Aws::String>
  CreateBrowserSettingsRequest& WithCustomerManagedKey(CustomerManagedKeyT&& value) {
    SetCustomerManagedKey(std::forward<CustomerManagedKeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Additional encryption context of the browser settings.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetAdditionalEncryptionContext() const { return m_additionalEncryptionContext; }
  inline bool AdditionalEncryptionContextHasBeenSet() const { return m_additionalEncryptionContextHasBeenSet; }
  template <typename AdditionalEncryptionContextT = Aws::Map<Aws::String, Aws::String>>
  void SetAdditionalEncryptionContext(AdditionalEncryptionContextT&& value) {
    m_additionalEncryptionContextHasBeenSet = true;
    m_additionalEncryptionContext = std::forward<AdditionalEncryptionContextT>(value);
  }
  template <typename AdditionalEncryptionContextT = Aws::Map<Aws::String, Aws::String>>
  CreateBrowserSettingsRequest& WithAdditionalEncryptionContext(AdditionalEncryptionContextT&& value) {
    SetAdditionalEncryptionContext(std::forward<AdditionalEncryptionContextT>(value));
    return *this;
  }
  template <typename AdditionalEncryptionContextKeyT = Aws::String, typename AdditionalEncryptionContextValueT = Aws::String>
  CreateBrowserSettingsRequest& AddAdditionalEncryptionContext(AdditionalEncryptionContextKeyT&& key,
                                                               AdditionalEncryptionContextValueT&& value) {
    m_additionalEncryptionContextHasBeenSet = true;
    m_additionalEncryptionContext.emplace(std::forward<AdditionalEncryptionContextKeyT>(key),
                                          std::forward<AdditionalEncryptionContextValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A JSON string containing Chrome Enterprise policies that will be applied to
   * all streaming sessions.</p>
   */
  inline const Aws::String& GetBrowserPolicy() const { return m_browserPolicy; }
  inline bool BrowserPolicyHasBeenSet() const { return m_browserPolicyHasBeenSet; }
  template <typename BrowserPolicyT = Aws::String>
  void SetBrowserPolicy(BrowserPolicyT&& value) {
    m_browserPolicyHasBeenSet = true;
    m_browserPolicy = std::forward<BrowserPolicyT>(value);
  }
  template <typename BrowserPolicyT = Aws::String>
  CreateBrowserSettingsRequest& WithBrowserPolicy(BrowserPolicyT&& value) {
    SetBrowserPolicy(std::forward<BrowserPolicyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. Idempotency ensures that an API request completes
   * only once. With an idempotent request, if the original request completes
   * successfully, subsequent retries with the same client token returns the result
   * from the original successful request.</p> <p>If you do not specify a client
   * token, one is automatically generated by the Amazon Web Services SDK. </p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateBrowserSettingsRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The policy that specifies which URLs end users are allowed to access or which
   * URLs or domain categories they are restricted from accessing for enhanced
   * security.</p>
   */
  inline const WebContentFilteringPolicy& GetWebContentFilteringPolicy() const { return m_webContentFilteringPolicy; }
  inline bool WebContentFilteringPolicyHasBeenSet() const { return m_webContentFilteringPolicyHasBeenSet; }
  template <typename WebContentFilteringPolicyT = WebContentFilteringPolicy>
  void SetWebContentFilteringPolicy(WebContentFilteringPolicyT&& value) {
    m_webContentFilteringPolicyHasBeenSet = true;
    m_webContentFilteringPolicy = std::forward<WebContentFilteringPolicyT>(value);
  }
  template <typename WebContentFilteringPolicyT = WebContentFilteringPolicy>
  CreateBrowserSettingsRequest& WithWebContentFilteringPolicy(WebContentFilteringPolicyT&& value) {
    SetWebContentFilteringPolicy(std::forward<WebContentFilteringPolicyT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Tag> m_tags;

  Aws::String m_customerManagedKey;

  Aws::Map<Aws::String, Aws::String> m_additionalEncryptionContext;

  Aws::String m_browserPolicy;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  WebContentFilteringPolicy m_webContentFilteringPolicy;
  bool m_tagsHasBeenSet = false;
  bool m_customerManagedKeyHasBeenSet = false;
  bool m_additionalEncryptionContextHasBeenSet = false;
  bool m_browserPolicyHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
  bool m_webContentFilteringPolicyHasBeenSet = false;
};

}  // namespace Model
}  // namespace WorkSpacesWeb
}  // namespace Aws
