﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/workmail/WorkMailRequest.h>
#include <aws/workmail/WorkMail_EXPORTS.h>
#include <aws/workmail/model/MobileDeviceAccessRuleEffect.h>

#include <utility>

namespace Aws {
namespace WorkMail {
namespace Model {

/**
 */
class PutMobileDeviceAccessOverrideRequest : public WorkMailRequest {
 public:
  AWS_WORKMAIL_API PutMobileDeviceAccessOverrideRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutMobileDeviceAccessOverride"; }

  AWS_WORKMAIL_API Aws::String SerializePayload() const override;

  AWS_WORKMAIL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Identifies the WorkMail organization for which you create the override.</p>
   */
  inline const Aws::String& GetOrganizationId() const { return m_organizationId; }
  inline bool OrganizationIdHasBeenSet() const { return m_organizationIdHasBeenSet; }
  template <typename OrganizationIdT = Aws::String>
  void SetOrganizationId(OrganizationIdT&& value) {
    m_organizationIdHasBeenSet = true;
    m_organizationId = std::forward<OrganizationIdT>(value);
  }
  template <typename OrganizationIdT = Aws::String>
  PutMobileDeviceAccessOverrideRequest& WithOrganizationId(OrganizationIdT&& value) {
    SetOrganizationId(std::forward<OrganizationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The WorkMail user for which you create the override. Accepts the following
   * types of user identities:</p> <ul> <li> <p>User ID:
   * <code>12345678-1234-1234-1234-123456789012</code> or
   * <code>S-1-1-12-1234567890-123456789-123456789-1234</code> </p> </li> <li>
   * <p>Email address: <code>user@domain.tld</code> </p> </li> <li> <p>User name:
   * <code>user</code> </p> </li> </ul>
   */
  inline const Aws::String& GetUserId() const { return m_userId; }
  inline bool UserIdHasBeenSet() const { return m_userIdHasBeenSet; }
  template <typename UserIdT = Aws::String>
  void SetUserId(UserIdT&& value) {
    m_userIdHasBeenSet = true;
    m_userId = std::forward<UserIdT>(value);
  }
  template <typename UserIdT = Aws::String>
  PutMobileDeviceAccessOverrideRequest& WithUserId(UserIdT&& value) {
    SetUserId(std::forward<UserIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The mobile device for which you create the override. <code>DeviceId</code> is
   * case insensitive.</p>
   */
  inline const Aws::String& GetDeviceId() const { return m_deviceId; }
  inline bool DeviceIdHasBeenSet() const { return m_deviceIdHasBeenSet; }
  template <typename DeviceIdT = Aws::String>
  void SetDeviceId(DeviceIdT&& value) {
    m_deviceIdHasBeenSet = true;
    m_deviceId = std::forward<DeviceIdT>(value);
  }
  template <typename DeviceIdT = Aws::String>
  PutMobileDeviceAccessOverrideRequest& WithDeviceId(DeviceIdT&& value) {
    SetDeviceId(std::forward<DeviceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The effect of the override, <code>ALLOW</code> or <code>DENY</code>.</p>
   */
  inline MobileDeviceAccessRuleEffect GetEffect() const { return m_effect; }
  inline bool EffectHasBeenSet() const { return m_effectHasBeenSet; }
  inline void SetEffect(MobileDeviceAccessRuleEffect value) {
    m_effectHasBeenSet = true;
    m_effect = value;
  }
  inline PutMobileDeviceAccessOverrideRequest& WithEffect(MobileDeviceAccessRuleEffect value) {
    SetEffect(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the override.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  PutMobileDeviceAccessOverrideRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_organizationId;

  Aws::String m_userId;

  Aws::String m_deviceId;

  MobileDeviceAccessRuleEffect m_effect{MobileDeviceAccessRuleEffect::NOT_SET};

  Aws::String m_description;
  bool m_organizationIdHasBeenSet = false;
  bool m_userIdHasBeenSet = false;
  bool m_deviceIdHasBeenSet = false;
  bool m_effectHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
};

}  // namespace Model
}  // namespace WorkMail
}  // namespace Aws
