﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/states/SFNRequest.h>
#include <aws/states/SFN_EXPORTS.h>
#include <aws/states/model/RoutingConfigurationListItem.h>

#include <utility>

namespace Aws {
namespace SFN {
namespace Model {

/**
 */
class UpdateStateMachineAliasRequest : public SFNRequest {
 public:
  AWS_SFN_API UpdateStateMachineAliasRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateStateMachineAlias"; }

  AWS_SFN_API Aws::String SerializePayload() const override;

  AWS_SFN_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the state machine alias.</p>
   */
  inline const Aws::String& GetStateMachineAliasArn() const { return m_stateMachineAliasArn; }
  inline bool StateMachineAliasArnHasBeenSet() const { return m_stateMachineAliasArnHasBeenSet; }
  template <typename StateMachineAliasArnT = Aws::String>
  void SetStateMachineAliasArn(StateMachineAliasArnT&& value) {
    m_stateMachineAliasArnHasBeenSet = true;
    m_stateMachineAliasArn = std::forward<StateMachineAliasArnT>(value);
  }
  template <typename StateMachineAliasArnT = Aws::String>
  UpdateStateMachineAliasRequest& WithStateMachineAliasArn(StateMachineAliasArnT&& value) {
    SetStateMachineAliasArn(std::forward<StateMachineAliasArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the state machine alias.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateStateMachineAliasRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The routing configuration of the state machine alias.</p> <p>An array of
   * <code>RoutingConfig</code> objects that specifies up to two state machine
   * versions that the alias starts executions for.</p>
   */
  inline const Aws::Vector<RoutingConfigurationListItem>& GetRoutingConfiguration() const { return m_routingConfiguration; }
  inline bool RoutingConfigurationHasBeenSet() const { return m_routingConfigurationHasBeenSet; }
  template <typename RoutingConfigurationT = Aws::Vector<RoutingConfigurationListItem>>
  void SetRoutingConfiguration(RoutingConfigurationT&& value) {
    m_routingConfigurationHasBeenSet = true;
    m_routingConfiguration = std::forward<RoutingConfigurationT>(value);
  }
  template <typename RoutingConfigurationT = Aws::Vector<RoutingConfigurationListItem>>
  UpdateStateMachineAliasRequest& WithRoutingConfiguration(RoutingConfigurationT&& value) {
    SetRoutingConfiguration(std::forward<RoutingConfigurationT>(value));
    return *this;
  }
  template <typename RoutingConfigurationT = RoutingConfigurationListItem>
  UpdateStateMachineAliasRequest& AddRoutingConfiguration(RoutingConfigurationT&& value) {
    m_routingConfigurationHasBeenSet = true;
    m_routingConfiguration.emplace_back(std::forward<RoutingConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_stateMachineAliasArn;

  Aws::String m_description;

  Aws::Vector<RoutingConfigurationListItem> m_routingConfiguration;
  bool m_stateMachineAliasArnHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_routingConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace SFN
}  // namespace Aws
