﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class UpdateExperimentRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API UpdateExperimentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateExperiment"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the experiment to update.</p>
   */
  inline const Aws::String& GetExperimentName() const { return m_experimentName; }
  inline bool ExperimentNameHasBeenSet() const { return m_experimentNameHasBeenSet; }
  template <typename ExperimentNameT = Aws::String>
  void SetExperimentName(ExperimentNameT&& value) {
    m_experimentNameHasBeenSet = true;
    m_experimentName = std::forward<ExperimentNameT>(value);
  }
  template <typename ExperimentNameT = Aws::String>
  UpdateExperimentRequest& WithExperimentName(ExperimentNameT&& value) {
    SetExperimentName(std::forward<ExperimentNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the experiment as displayed. The name doesn't need to be unique.
   * If <code>DisplayName</code> isn't specified, <code>ExperimentName</code> is
   * displayed.</p>
   */
  inline const Aws::String& GetDisplayName() const { return m_displayName; }
  inline bool DisplayNameHasBeenSet() const { return m_displayNameHasBeenSet; }
  template <typename DisplayNameT = Aws::String>
  void SetDisplayName(DisplayNameT&& value) {
    m_displayNameHasBeenSet = true;
    m_displayName = std::forward<DisplayNameT>(value);
  }
  template <typename DisplayNameT = Aws::String>
  UpdateExperimentRequest& WithDisplayName(DisplayNameT&& value) {
    SetDisplayName(std::forward<DisplayNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the experiment.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateExperimentRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_experimentName;

  Aws::String m_displayName;

  Aws::String m_description;
  bool m_experimentNameHasBeenSet = false;
  bool m_displayNameHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
