﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/redshift/RedshiftRequest.h>
#include <aws/redshift/Redshift_EXPORTS.h>
#include <aws/redshift/model/Tag.h>

#include <utility>

namespace Aws {
namespace Redshift {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/redshift-2012-12-01/CreateClusterSnapshotMessage">AWS
 * API Reference</a></p>
 */
class CreateClusterSnapshotRequest : public RedshiftRequest {
 public:
  AWS_REDSHIFT_API CreateClusterSnapshotRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateClusterSnapshot"; }

  AWS_REDSHIFT_API Aws::String SerializePayload() const override;

 protected:
  AWS_REDSHIFT_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>A unique identifier for the snapshot that you are requesting. This identifier
   * must be unique for all snapshots within the Amazon Web Services account.</p>
   * <p>Constraints:</p> <ul> <li> <p>Cannot be null, empty, or blank</p> </li> <li>
   * <p>Must contain from 1 to 255 alphanumeric characters or hyphens</p> </li> <li>
   * <p>First character must be a letter</p> </li> <li> <p>Cannot end with a hyphen
   * or contain two consecutive hyphens</p> </li> </ul> <p>Example:
   * <code>my-snapshot-id</code> </p>
   */
  inline const Aws::String& GetSnapshotIdentifier() const { return m_snapshotIdentifier; }
  inline bool SnapshotIdentifierHasBeenSet() const { return m_snapshotIdentifierHasBeenSet; }
  template <typename SnapshotIdentifierT = Aws::String>
  void SetSnapshotIdentifier(SnapshotIdentifierT&& value) {
    m_snapshotIdentifierHasBeenSet = true;
    m_snapshotIdentifier = std::forward<SnapshotIdentifierT>(value);
  }
  template <typename SnapshotIdentifierT = Aws::String>
  CreateClusterSnapshotRequest& WithSnapshotIdentifier(SnapshotIdentifierT&& value) {
    SetSnapshotIdentifier(std::forward<SnapshotIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The cluster identifier for which you want a snapshot.</p>
   */
  inline const Aws::String& GetClusterIdentifier() const { return m_clusterIdentifier; }
  inline bool ClusterIdentifierHasBeenSet() const { return m_clusterIdentifierHasBeenSet; }
  template <typename ClusterIdentifierT = Aws::String>
  void SetClusterIdentifier(ClusterIdentifierT&& value) {
    m_clusterIdentifierHasBeenSet = true;
    m_clusterIdentifier = std::forward<ClusterIdentifierT>(value);
  }
  template <typename ClusterIdentifierT = Aws::String>
  CreateClusterSnapshotRequest& WithClusterIdentifier(ClusterIdentifierT&& value) {
    SetClusterIdentifier(std::forward<ClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of days that a manual snapshot is retained. If the value is -1,
   * the manual snapshot is retained indefinitely. </p> <p>The value must be either
   * -1 or an integer between 1 and 3,653.</p> <p>The default value is -1.</p>
   */
  inline int GetManualSnapshotRetentionPeriod() const { return m_manualSnapshotRetentionPeriod; }
  inline bool ManualSnapshotRetentionPeriodHasBeenSet() const { return m_manualSnapshotRetentionPeriodHasBeenSet; }
  inline void SetManualSnapshotRetentionPeriod(int value) {
    m_manualSnapshotRetentionPeriodHasBeenSet = true;
    m_manualSnapshotRetentionPeriod = value;
  }
  inline CreateClusterSnapshotRequest& WithManualSnapshotRetentionPeriod(int value) {
    SetManualSnapshotRetentionPeriod(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tag instances.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateClusterSnapshotRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateClusterSnapshotRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_snapshotIdentifier;

  Aws::String m_clusterIdentifier;

  int m_manualSnapshotRetentionPeriod{0};

  Aws::Vector<Tag> m_tags;
  bool m_snapshotIdentifierHasBeenSet = false;
  bool m_clusterIdentifierHasBeenSet = false;
  bool m_manualSnapshotRetentionPeriodHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Redshift
}  // namespace Aws
