﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/redshift/Redshift_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace Redshift {
namespace Model {

/**
 * <p>Returns the destination region and retention period that are configured for
 * cross-region snapshot copy.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/redshift-2012-12-01/ClusterSnapshotCopyStatus">AWS
 * API Reference</a></p>
 */
class ClusterSnapshotCopyStatus {
 public:
  AWS_REDSHIFT_API ClusterSnapshotCopyStatus() = default;
  AWS_REDSHIFT_API ClusterSnapshotCopyStatus(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_REDSHIFT_API ClusterSnapshotCopyStatus& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_REDSHIFT_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_REDSHIFT_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The destination region that snapshots are automatically copied to when
   * cross-region snapshot copy is enabled.</p>
   */
  inline const Aws::String& GetDestinationRegion() const { return m_destinationRegion; }
  inline bool DestinationRegionHasBeenSet() const { return m_destinationRegionHasBeenSet; }
  template <typename DestinationRegionT = Aws::String>
  void SetDestinationRegion(DestinationRegionT&& value) {
    m_destinationRegionHasBeenSet = true;
    m_destinationRegion = std::forward<DestinationRegionT>(value);
  }
  template <typename DestinationRegionT = Aws::String>
  ClusterSnapshotCopyStatus& WithDestinationRegion(DestinationRegionT&& value) {
    SetDestinationRegion(std::forward<DestinationRegionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of days that automated snapshots are retained in the destination
   * region after they are copied from a source region.</p>
   */
  inline long long GetRetentionPeriod() const { return m_retentionPeriod; }
  inline bool RetentionPeriodHasBeenSet() const { return m_retentionPeriodHasBeenSet; }
  inline void SetRetentionPeriod(long long value) {
    m_retentionPeriodHasBeenSet = true;
    m_retentionPeriod = value;
  }
  inline ClusterSnapshotCopyStatus& WithRetentionPeriod(long long value) {
    SetRetentionPeriod(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of days that automated snapshots are retained in the destination
   * region after they are copied from a source region. If the value is -1, the
   * manual snapshot is retained indefinitely. </p> <p>The value must be either -1 or
   * an integer between 1 and 3,653.</p>
   */
  inline int GetManualSnapshotRetentionPeriod() const { return m_manualSnapshotRetentionPeriod; }
  inline bool ManualSnapshotRetentionPeriodHasBeenSet() const { return m_manualSnapshotRetentionPeriodHasBeenSet; }
  inline void SetManualSnapshotRetentionPeriod(int value) {
    m_manualSnapshotRetentionPeriodHasBeenSet = true;
    m_manualSnapshotRetentionPeriod = value;
  }
  inline ClusterSnapshotCopyStatus& WithManualSnapshotRetentionPeriod(int value) {
    SetManualSnapshotRetentionPeriod(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the snapshot copy grant.</p>
   */
  inline const Aws::String& GetSnapshotCopyGrantName() const { return m_snapshotCopyGrantName; }
  inline bool SnapshotCopyGrantNameHasBeenSet() const { return m_snapshotCopyGrantNameHasBeenSet; }
  template <typename SnapshotCopyGrantNameT = Aws::String>
  void SetSnapshotCopyGrantName(SnapshotCopyGrantNameT&& value) {
    m_snapshotCopyGrantNameHasBeenSet = true;
    m_snapshotCopyGrantName = std::forward<SnapshotCopyGrantNameT>(value);
  }
  template <typename SnapshotCopyGrantNameT = Aws::String>
  ClusterSnapshotCopyStatus& WithSnapshotCopyGrantName(SnapshotCopyGrantNameT&& value) {
    SetSnapshotCopyGrantName(std::forward<SnapshotCopyGrantNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_destinationRegion;

  long long m_retentionPeriod{0};

  int m_manualSnapshotRetentionPeriod{0};

  Aws::String m_snapshotCopyGrantName;
  bool m_destinationRegionHasBeenSet = false;
  bool m_retentionPeriodHasBeenSet = false;
  bool m_manualSnapshotRetentionPeriodHasBeenSet = false;
  bool m_snapshotCopyGrantNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace Redshift
}  // namespace Aws
