﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/personalize/Personalize_EXPORTS.h>
#include <aws/personalize/model/DatasetExportJobOutput.h>
#include <aws/personalize/model/IngestionMode.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Personalize {
namespace Model {

/**
 * <p>Describes a job that exports a dataset to an Amazon S3 bucket. For more
 * information, see <a
 * href="https://docs.aws.amazon.com/personalize/latest/dg/API_CreateDatasetExportJob.html">CreateDatasetExportJob</a>.</p>
 * <p>A dataset export job can be in one of the following states:</p> <ul> <li>
 * <p>CREATE PENDING &gt; CREATE IN_PROGRESS &gt; ACTIVE -or- CREATE FAILED</p>
 * </li> </ul><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/personalize-2018-05-22/DatasetExportJob">AWS
 * API Reference</a></p>
 */
class DatasetExportJob {
 public:
  AWS_PERSONALIZE_API DatasetExportJob() = default;
  AWS_PERSONALIZE_API DatasetExportJob(Aws::Utils::Json::JsonView jsonValue);
  AWS_PERSONALIZE_API DatasetExportJob& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_PERSONALIZE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the export job.</p>
   */
  inline const Aws::String& GetJobName() const { return m_jobName; }
  inline bool JobNameHasBeenSet() const { return m_jobNameHasBeenSet; }
  template <typename JobNameT = Aws::String>
  void SetJobName(JobNameT&& value) {
    m_jobNameHasBeenSet = true;
    m_jobName = std::forward<JobNameT>(value);
  }
  template <typename JobNameT = Aws::String>
  DatasetExportJob& WithJobName(JobNameT&& value) {
    SetJobName(std::forward<JobNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the dataset export job.</p>
   */
  inline const Aws::String& GetDatasetExportJobArn() const { return m_datasetExportJobArn; }
  inline bool DatasetExportJobArnHasBeenSet() const { return m_datasetExportJobArnHasBeenSet; }
  template <typename DatasetExportJobArnT = Aws::String>
  void SetDatasetExportJobArn(DatasetExportJobArnT&& value) {
    m_datasetExportJobArnHasBeenSet = true;
    m_datasetExportJobArn = std::forward<DatasetExportJobArnT>(value);
  }
  template <typename DatasetExportJobArnT = Aws::String>
  DatasetExportJob& WithDatasetExportJobArn(DatasetExportJobArnT&& value) {
    SetDatasetExportJobArn(std::forward<DatasetExportJobArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the dataset to export.</p>
   */
  inline const Aws::String& GetDatasetArn() const { return m_datasetArn; }
  inline bool DatasetArnHasBeenSet() const { return m_datasetArnHasBeenSet; }
  template <typename DatasetArnT = Aws::String>
  void SetDatasetArn(DatasetArnT&& value) {
    m_datasetArnHasBeenSet = true;
    m_datasetArn = std::forward<DatasetArnT>(value);
  }
  template <typename DatasetArnT = Aws::String>
  DatasetExportJob& WithDatasetArn(DatasetArnT&& value) {
    SetDatasetArn(std::forward<DatasetArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The data to export, based on how you imported the data. You can choose to
   * export <code>BULK</code> data that you imported using a dataset import job,
   * <code>PUT</code> data that you imported incrementally (using the console,
   * PutEvents, PutUsers and PutItems operations), or <code>ALL</code> for both
   * types. The default value is <code>PUT</code>. </p>
   */
  inline IngestionMode GetIngestionMode() const { return m_ingestionMode; }
  inline bool IngestionModeHasBeenSet() const { return m_ingestionModeHasBeenSet; }
  inline void SetIngestionMode(IngestionMode value) {
    m_ingestionModeHasBeenSet = true;
    m_ingestionMode = value;
  }
  inline DatasetExportJob& WithIngestionMode(IngestionMode value) {
    SetIngestionMode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the IAM service role that has permissions
   * to add data to your output Amazon S3 bucket.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  DatasetExportJob& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the dataset export job.</p> <p>A dataset export job can be in
   * one of the following states:</p> <ul> <li> <p>CREATE PENDING &gt; CREATE
   * IN_PROGRESS &gt; ACTIVE -or- CREATE FAILED</p> </li> </ul>
   */
  inline const Aws::String& GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  template <typename StatusT = Aws::String>
  void SetStatus(StatusT&& value) {
    m_statusHasBeenSet = true;
    m_status = std::forward<StatusT>(value);
  }
  template <typename StatusT = Aws::String>
  DatasetExportJob& WithStatus(StatusT&& value) {
    SetStatus(std::forward<StatusT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The path to the Amazon S3 bucket where the job's output is stored. For
   * example:</p> <p> <code>s3://bucket-name/folder-name/</code> </p>
   */
  inline const DatasetExportJobOutput& GetJobOutput() const { return m_jobOutput; }
  inline bool JobOutputHasBeenSet() const { return m_jobOutputHasBeenSet; }
  template <typename JobOutputT = DatasetExportJobOutput>
  void SetJobOutput(JobOutputT&& value) {
    m_jobOutputHasBeenSet = true;
    m_jobOutput = std::forward<JobOutputT>(value);
  }
  template <typename JobOutputT = DatasetExportJobOutput>
  DatasetExportJob& WithJobOutput(JobOutputT&& value) {
    SetJobOutput(std::forward<JobOutputT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The creation date and time (in Unix time) of the dataset export job.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationDateTime() const { return m_creationDateTime; }
  inline bool CreationDateTimeHasBeenSet() const { return m_creationDateTimeHasBeenSet; }
  template <typename CreationDateTimeT = Aws::Utils::DateTime>
  void SetCreationDateTime(CreationDateTimeT&& value) {
    m_creationDateTimeHasBeenSet = true;
    m_creationDateTime = std::forward<CreationDateTimeT>(value);
  }
  template <typename CreationDateTimeT = Aws::Utils::DateTime>
  DatasetExportJob& WithCreationDateTime(CreationDateTimeT&& value) {
    SetCreationDateTime(std::forward<CreationDateTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time (in Unix time) the status of the dataset export job was
   * last updated.</p>
   */
  inline const Aws::Utils::DateTime& GetLastUpdatedDateTime() const { return m_lastUpdatedDateTime; }
  inline bool LastUpdatedDateTimeHasBeenSet() const { return m_lastUpdatedDateTimeHasBeenSet; }
  template <typename LastUpdatedDateTimeT = Aws::Utils::DateTime>
  void SetLastUpdatedDateTime(LastUpdatedDateTimeT&& value) {
    m_lastUpdatedDateTimeHasBeenSet = true;
    m_lastUpdatedDateTime = std::forward<LastUpdatedDateTimeT>(value);
  }
  template <typename LastUpdatedDateTimeT = Aws::Utils::DateTime>
  DatasetExportJob& WithLastUpdatedDateTime(LastUpdatedDateTimeT&& value) {
    SetLastUpdatedDateTime(std::forward<LastUpdatedDateTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If a dataset export job fails, provides the reason why.</p>
   */
  inline const Aws::String& GetFailureReason() const { return m_failureReason; }
  inline bool FailureReasonHasBeenSet() const { return m_failureReasonHasBeenSet; }
  template <typename FailureReasonT = Aws::String>
  void SetFailureReason(FailureReasonT&& value) {
    m_failureReasonHasBeenSet = true;
    m_failureReason = std::forward<FailureReasonT>(value);
  }
  template <typename FailureReasonT = Aws::String>
  DatasetExportJob& WithFailureReason(FailureReasonT&& value) {
    SetFailureReason(std::forward<FailureReasonT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_jobName;

  Aws::String m_datasetExportJobArn;

  Aws::String m_datasetArn;

  IngestionMode m_ingestionMode{IngestionMode::NOT_SET};

  Aws::String m_roleArn;

  Aws::String m_status;

  DatasetExportJobOutput m_jobOutput;

  Aws::Utils::DateTime m_creationDateTime{};

  Aws::Utils::DateTime m_lastUpdatedDateTime{};

  Aws::String m_failureReason;
  bool m_jobNameHasBeenSet = false;
  bool m_datasetExportJobArnHasBeenSet = false;
  bool m_datasetArnHasBeenSet = false;
  bool m_ingestionModeHasBeenSet = false;
  bool m_roleArnHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_jobOutputHasBeenSet = false;
  bool m_creationDateTimeHasBeenSet = false;
  bool m_lastUpdatedDateTimeHasBeenSet = false;
  bool m_failureReasonHasBeenSet = false;
};

}  // namespace Model
}  // namespace Personalize
}  // namespace Aws
