﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/managedblockchain/ManagedBlockchainRequest.h>
#include <aws/managedblockchain/ManagedBlockchain_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace ManagedBlockchain {
namespace Model {

/**
 */
class GetNodeRequest : public ManagedBlockchainRequest {
 public:
  AWS_MANAGEDBLOCKCHAIN_API GetNodeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetNode"; }

  AWS_MANAGEDBLOCKCHAIN_API Aws::String SerializePayload() const override;

  AWS_MANAGEDBLOCKCHAIN_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The unique identifier of the network that the node is on.</p>
   */
  inline const Aws::String& GetNetworkId() const { return m_networkId; }
  inline bool NetworkIdHasBeenSet() const { return m_networkIdHasBeenSet; }
  template <typename NetworkIdT = Aws::String>
  void SetNetworkId(NetworkIdT&& value) {
    m_networkIdHasBeenSet = true;
    m_networkId = std::forward<NetworkIdT>(value);
  }
  template <typename NetworkIdT = Aws::String>
  GetNodeRequest& WithNetworkId(NetworkIdT&& value) {
    SetNetworkId(std::forward<NetworkIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the member that owns the node.</p> <p>Applies only
   * to Hyperledger Fabric and is required for Hyperledger Fabric.</p>
   */
  inline const Aws::String& GetMemberId() const { return m_memberId; }
  inline bool MemberIdHasBeenSet() const { return m_memberIdHasBeenSet; }
  template <typename MemberIdT = Aws::String>
  void SetMemberId(MemberIdT&& value) {
    m_memberIdHasBeenSet = true;
    m_memberId = std::forward<MemberIdT>(value);
  }
  template <typename MemberIdT = Aws::String>
  GetNodeRequest& WithMemberId(MemberIdT&& value) {
    SetMemberId(std::forward<MemberIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the node.</p>
   */
  inline const Aws::String& GetNodeId() const { return m_nodeId; }
  inline bool NodeIdHasBeenSet() const { return m_nodeIdHasBeenSet; }
  template <typename NodeIdT = Aws::String>
  void SetNodeId(NodeIdT&& value) {
    m_nodeIdHasBeenSet = true;
    m_nodeId = std::forward<NodeIdT>(value);
  }
  template <typename NodeIdT = Aws::String>
  GetNodeRequest& WithNodeId(NodeIdT&& value) {
    SetNodeId(std::forward<NodeIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_networkId;

  Aws::String m_memberId;

  Aws::String m_nodeId;
  bool m_networkIdHasBeenSet = false;
  bool m_memberIdHasBeenSet = false;
  bool m_nodeIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace ManagedBlockchain
}  // namespace Aws
