﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iotwireless/IoTWirelessRequest.h>
#include <aws/iotwireless/IoTWireless_EXPORTS.h>
#include <aws/iotwireless/model/PartnerType.h>
#include <aws/iotwireless/model/SidewalkUpdateAccount.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace IoTWireless {
namespace Model {

/**
 */
class UpdatePartnerAccountRequest : public IoTWirelessRequest {
 public:
  AWS_IOTWIRELESS_API UpdatePartnerAccountRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdatePartnerAccount"; }

  AWS_IOTWIRELESS_API Aws::String SerializePayload() const override;

  AWS_IOTWIRELESS_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The Sidewalk account credentials.</p>
   */
  inline const SidewalkUpdateAccount& GetSidewalk() const { return m_sidewalk; }
  inline bool SidewalkHasBeenSet() const { return m_sidewalkHasBeenSet; }
  template <typename SidewalkT = SidewalkUpdateAccount>
  void SetSidewalk(SidewalkT&& value) {
    m_sidewalkHasBeenSet = true;
    m_sidewalk = std::forward<SidewalkT>(value);
  }
  template <typename SidewalkT = SidewalkUpdateAccount>
  UpdatePartnerAccountRequest& WithSidewalk(SidewalkT&& value) {
    SetSidewalk(std::forward<SidewalkT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the partner account to update.</p>
   */
  inline const Aws::String& GetPartnerAccountId() const { return m_partnerAccountId; }
  inline bool PartnerAccountIdHasBeenSet() const { return m_partnerAccountIdHasBeenSet; }
  template <typename PartnerAccountIdT = Aws::String>
  void SetPartnerAccountId(PartnerAccountIdT&& value) {
    m_partnerAccountIdHasBeenSet = true;
    m_partnerAccountId = std::forward<PartnerAccountIdT>(value);
  }
  template <typename PartnerAccountIdT = Aws::String>
  UpdatePartnerAccountRequest& WithPartnerAccountId(PartnerAccountIdT&& value) {
    SetPartnerAccountId(std::forward<PartnerAccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The partner type.</p>
   */
  inline PartnerType GetPartnerType() const { return m_partnerType; }
  inline bool PartnerTypeHasBeenSet() const { return m_partnerTypeHasBeenSet; }
  inline void SetPartnerType(PartnerType value) {
    m_partnerTypeHasBeenSet = true;
    m_partnerType = value;
  }
  inline UpdatePartnerAccountRequest& WithPartnerType(PartnerType value) {
    SetPartnerType(value);
    return *this;
  }
  ///@}
 private:
  SidewalkUpdateAccount m_sidewalk;

  Aws::String m_partnerAccountId;

  PartnerType m_partnerType{PartnerType::NOT_SET};
  bool m_sidewalkHasBeenSet = false;
  bool m_partnerAccountIdHasBeenSet = false;
  bool m_partnerTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTWireless
}  // namespace Aws
