﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iot/IoTRequest.h>
#include <aws/iot/IoT_EXPORTS.h>
#include <aws/iot/model/ThingPrincipalType.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace IoT {
namespace Model {

/**
 */
class ListPrincipalThingsV2Request : public IoTRequest {
 public:
  AWS_IOT_API ListPrincipalThingsV2Request() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListPrincipalThingsV2"; }

  AWS_IOT_API Aws::String SerializePayload() const override;

  AWS_IOT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  AWS_IOT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>To retrieve the next set of results, the <code>nextToken</code> value from a
   * previous response; otherwise <b>null</b> to receive the first set of
   * results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListPrincipalThingsV2Request& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return in this operation.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListPrincipalThingsV2Request& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The principal. A principal can be an X.509 certificate or an Amazon Cognito
   * ID.</p>
   */
  inline const Aws::String& GetPrincipal() const { return m_principal; }
  inline bool PrincipalHasBeenSet() const { return m_principalHasBeenSet; }
  template <typename PrincipalT = Aws::String>
  void SetPrincipal(PrincipalT&& value) {
    m_principalHasBeenSet = true;
    m_principal = std::forward<PrincipalT>(value);
  }
  template <typename PrincipalT = Aws::String>
  ListPrincipalThingsV2Request& WithPrincipal(PrincipalT&& value) {
    SetPrincipal(std::forward<PrincipalT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of the relation you want to filter in the response. If no value is
   * provided in this field, the response will list all things, including both the
   * <code>EXCLUSIVE_THING</code> and <code>NON_EXCLUSIVE_THING</code> attachment
   * types.</p> <ul> <li> <p> <code>EXCLUSIVE_THING</code> - Attaches the specified
   * principal to the specified thing, exclusively. The thing will be the only thing
   * that’s attached to the principal.</p> </li> </ul> <ul> <li> <p>
   * <code>NON_EXCLUSIVE_THING</code> - Attaches the specified principal to the
   * specified thing. Multiple things can be attached to the principal.</p> </li>
   * </ul>
   */
  inline ThingPrincipalType GetThingPrincipalType() const { return m_thingPrincipalType; }
  inline bool ThingPrincipalTypeHasBeenSet() const { return m_thingPrincipalTypeHasBeenSet; }
  inline void SetThingPrincipalType(ThingPrincipalType value) {
    m_thingPrincipalTypeHasBeenSet = true;
    m_thingPrincipalType = value;
  }
  inline ListPrincipalThingsV2Request& WithThingPrincipalType(ThingPrincipalType value) {
    SetThingPrincipalType(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_nextToken;

  int m_maxResults{0};

  Aws::String m_principal;

  ThingPrincipalType m_thingPrincipalType{ThingPrincipalType::NOT_SET};
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_principalHasBeenSet = false;
  bool m_thingPrincipalTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoT
}  // namespace Aws
