﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iot/IoTRequest.h>
#include <aws/iot/IoT_EXPORTS.h>
#include <aws/iot/model/Sbom.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace IoT {
namespace Model {

/**
 */
class AssociateSbomWithPackageVersionRequest : public IoTRequest {
 public:
  AWS_IOT_API AssociateSbomWithPackageVersionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AssociateSbomWithPackageVersion"; }

  AWS_IOT_API Aws::String SerializePayload() const override;

  AWS_IOT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The name of the new software package.</p>
   */
  inline const Aws::String& GetPackageName() const { return m_packageName; }
  inline bool PackageNameHasBeenSet() const { return m_packageNameHasBeenSet; }
  template <typename PackageNameT = Aws::String>
  void SetPackageName(PackageNameT&& value) {
    m_packageNameHasBeenSet = true;
    m_packageName = std::forward<PackageNameT>(value);
  }
  template <typename PackageNameT = Aws::String>
  AssociateSbomWithPackageVersionRequest& WithPackageName(PackageNameT&& value) {
    SetPackageName(std::forward<PackageNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the new package version.</p>
   */
  inline const Aws::String& GetVersionName() const { return m_versionName; }
  inline bool VersionNameHasBeenSet() const { return m_versionNameHasBeenSet; }
  template <typename VersionNameT = Aws::String>
  void SetVersionName(VersionNameT&& value) {
    m_versionNameHasBeenSet = true;
    m_versionName = std::forward<VersionNameT>(value);
  }
  template <typename VersionNameT = Aws::String>
  AssociateSbomWithPackageVersionRequest& WithVersionName(VersionNameT&& value) {
    SetVersionName(std::forward<VersionNameT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Sbom& GetSbom() const { return m_sbom; }
  inline bool SbomHasBeenSet() const { return m_sbomHasBeenSet; }
  template <typename SbomT = Sbom>
  void SetSbom(SbomT&& value) {
    m_sbomHasBeenSet = true;
    m_sbom = std::forward<SbomT>(value);
  }
  template <typename SbomT = Sbom>
  AssociateSbomWithPackageVersionRequest& WithSbom(SbomT&& value) {
    SetSbom(std::forward<SbomT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique case-sensitive identifier that you can provide to ensure the
   * idempotency of the request. Don't reuse this client token if a new idempotent
   * request is required.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  AssociateSbomWithPackageVersionRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_packageName;

  Aws::String m_versionName;

  Sbom m_sbom;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_packageNameHasBeenSet = false;
  bool m_versionNameHasBeenSet = false;
  bool m_sbomHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace IoT
}  // namespace Aws
