﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/eventbridge/EventBridgeRequest.h>
#include <aws/eventbridge/EventBridge_EXPORTS.h>

#include <utility>

namespace Aws {
namespace EventBridge {
namespace Model {

/**
 */
class CreateArchiveRequest : public EventBridgeRequest {
 public:
  AWS_EVENTBRIDGE_API CreateArchiveRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateArchive"; }

  AWS_EVENTBRIDGE_API Aws::String SerializePayload() const override;

  AWS_EVENTBRIDGE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name for the archive to create.</p>
   */
  inline const Aws::String& GetArchiveName() const { return m_archiveName; }
  inline bool ArchiveNameHasBeenSet() const { return m_archiveNameHasBeenSet; }
  template <typename ArchiveNameT = Aws::String>
  void SetArchiveName(ArchiveNameT&& value) {
    m_archiveNameHasBeenSet = true;
    m_archiveName = std::forward<ArchiveNameT>(value);
  }
  template <typename ArchiveNameT = Aws::String>
  CreateArchiveRequest& WithArchiveName(ArchiveNameT&& value) {
    SetArchiveName(std::forward<ArchiveNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the event bus that sends events to the archive.</p>
   */
  inline const Aws::String& GetEventSourceArn() const { return m_eventSourceArn; }
  inline bool EventSourceArnHasBeenSet() const { return m_eventSourceArnHasBeenSet; }
  template <typename EventSourceArnT = Aws::String>
  void SetEventSourceArn(EventSourceArnT&& value) {
    m_eventSourceArnHasBeenSet = true;
    m_eventSourceArn = std::forward<EventSourceArnT>(value);
  }
  template <typename EventSourceArnT = Aws::String>
  CreateArchiveRequest& WithEventSourceArn(EventSourceArnT&& value) {
    SetEventSourceArn(std::forward<EventSourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description for the archive.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateArchiveRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An event pattern to use to filter events sent to the archive.</p>
   */
  inline const Aws::String& GetEventPattern() const { return m_eventPattern; }
  inline bool EventPatternHasBeenSet() const { return m_eventPatternHasBeenSet; }
  template <typename EventPatternT = Aws::String>
  void SetEventPattern(EventPatternT&& value) {
    m_eventPatternHasBeenSet = true;
    m_eventPattern = std::forward<EventPatternT>(value);
  }
  template <typename EventPatternT = Aws::String>
  CreateArchiveRequest& WithEventPattern(EventPatternT&& value) {
    SetEventPattern(std::forward<EventPatternT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of days to retain events for. Default value is 0. If set to 0,
   * events are retained indefinitely</p>
   */
  inline int GetRetentionDays() const { return m_retentionDays; }
  inline bool RetentionDaysHasBeenSet() const { return m_retentionDaysHasBeenSet; }
  inline void SetRetentionDays(int value) {
    m_retentionDaysHasBeenSet = true;
    m_retentionDays = value;
  }
  inline CreateArchiveRequest& WithRetentionDays(int value) {
    SetRetentionDays(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the KMS customer managed key for EventBridge to use, if you
   * choose to use a customer managed key to encrypt this archive. The identifier can
   * be the key Amazon Resource Name (ARN), KeyId, key alias, or key alias ARN.</p>
   * <p>If you do not specify a customer managed key identifier, EventBridge uses an
   * Amazon Web Services owned key to encrypt the archive.</p> <p>For more
   * information, see <a
   * href="https://docs.aws.amazon.com/kms/latest/developerguide/viewing-keys.html">Identify
   * and view keys</a> in the <i>Key Management Service Developer Guide</i>. </p>
   *  <p>If you have specified that EventBridge use a customer managed key
   * for encrypting the source event bus, we strongly recommend you also specify a
   * customer managed key for any archives for the event bus as well. </p> <p>For
   * more information, see <a
   * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/encryption-archives.html">Encrypting
   * archives</a> in the <i>Amazon EventBridge User Guide</i>.</p>
   */
  inline const Aws::String& GetKmsKeyIdentifier() const { return m_kmsKeyIdentifier; }
  inline bool KmsKeyIdentifierHasBeenSet() const { return m_kmsKeyIdentifierHasBeenSet; }
  template <typename KmsKeyIdentifierT = Aws::String>
  void SetKmsKeyIdentifier(KmsKeyIdentifierT&& value) {
    m_kmsKeyIdentifierHasBeenSet = true;
    m_kmsKeyIdentifier = std::forward<KmsKeyIdentifierT>(value);
  }
  template <typename KmsKeyIdentifierT = Aws::String>
  CreateArchiveRequest& WithKmsKeyIdentifier(KmsKeyIdentifierT&& value) {
    SetKmsKeyIdentifier(std::forward<KmsKeyIdentifierT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_archiveName;

  Aws::String m_eventSourceArn;

  Aws::String m_description;

  Aws::String m_eventPattern;

  int m_retentionDays{0};

  Aws::String m_kmsKeyIdentifier;
  bool m_archiveNameHasBeenSet = false;
  bool m_eventSourceArnHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_eventPatternHasBeenSet = false;
  bool m_retentionDaysHasBeenSet = false;
  bool m_kmsKeyIdentifierHasBeenSet = false;
};

}  // namespace Model
}  // namespace EventBridge
}  // namespace Aws
