﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticbeanstalk/ElasticBeanstalkRequest.h>
#include <aws/elasticbeanstalk/ElasticBeanstalk_EXPORTS.h>
#include <aws/elasticbeanstalk/model/InstancesHealthAttribute.h>

#include <utility>

namespace Aws {
namespace ElasticBeanstalk {
namespace Model {

/**
 * <p>Parameters for a call to <code>DescribeInstancesHealth</code>.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticbeanstalk-2010-12-01/DescribeInstancesHealthRequest">AWS
 * API Reference</a></p>
 */
class DescribeInstancesHealthRequest : public ElasticBeanstalkRequest {
 public:
  AWS_ELASTICBEANSTALK_API DescribeInstancesHealthRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeInstancesHealth"; }

  AWS_ELASTICBEANSTALK_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICBEANSTALK_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>Specify the AWS Elastic Beanstalk environment by name.</p>
   */
  inline const Aws::String& GetEnvironmentName() const { return m_environmentName; }
  inline bool EnvironmentNameHasBeenSet() const { return m_environmentNameHasBeenSet; }
  template <typename EnvironmentNameT = Aws::String>
  void SetEnvironmentName(EnvironmentNameT&& value) {
    m_environmentNameHasBeenSet = true;
    m_environmentName = std::forward<EnvironmentNameT>(value);
  }
  template <typename EnvironmentNameT = Aws::String>
  DescribeInstancesHealthRequest& WithEnvironmentName(EnvironmentNameT&& value) {
    SetEnvironmentName(std::forward<EnvironmentNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify the AWS Elastic Beanstalk environment by ID.</p>
   */
  inline const Aws::String& GetEnvironmentId() const { return m_environmentId; }
  inline bool EnvironmentIdHasBeenSet() const { return m_environmentIdHasBeenSet; }
  template <typename EnvironmentIdT = Aws::String>
  void SetEnvironmentId(EnvironmentIdT&& value) {
    m_environmentIdHasBeenSet = true;
    m_environmentId = std::forward<EnvironmentIdT>(value);
  }
  template <typename EnvironmentIdT = Aws::String>
  DescribeInstancesHealthRequest& WithEnvironmentId(EnvironmentIdT&& value) {
    SetEnvironmentId(std::forward<EnvironmentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the response elements you wish to receive. To retrieve all
   * attributes, set to <code>All</code>. If no attribute names are specified,
   * returns a list of instances.</p>
   */
  inline const Aws::Vector<InstancesHealthAttribute>& GetAttributeNames() const { return m_attributeNames; }
  inline bool AttributeNamesHasBeenSet() const { return m_attributeNamesHasBeenSet; }
  template <typename AttributeNamesT = Aws::Vector<InstancesHealthAttribute>>
  void SetAttributeNames(AttributeNamesT&& value) {
    m_attributeNamesHasBeenSet = true;
    m_attributeNames = std::forward<AttributeNamesT>(value);
  }
  template <typename AttributeNamesT = Aws::Vector<InstancesHealthAttribute>>
  DescribeInstancesHealthRequest& WithAttributeNames(AttributeNamesT&& value) {
    SetAttributeNames(std::forward<AttributeNamesT>(value));
    return *this;
  }
  inline DescribeInstancesHealthRequest& AddAttributeNames(InstancesHealthAttribute value) {
    m_attributeNamesHasBeenSet = true;
    m_attributeNames.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify the pagination token returned by a previous call.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribeInstancesHealthRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_environmentName;

  Aws::String m_environmentId;

  Aws::Vector<InstancesHealthAttribute> m_attributeNames;

  Aws::String m_nextToken;
  bool m_environmentNameHasBeenSet = false;
  bool m_environmentIdHasBeenSet = false;
  bool m_attributeNamesHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticBeanstalk
}  // namespace Aws
