﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ecs/ECS_EXPORTS.h>
#include <aws/ecs/model/NetworkBinding.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace ECS {
namespace Model {

/**
 * <p>An object that represents a change in state for a container.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/ContainerStateChange">AWS
 * API Reference</a></p>
 */
class ContainerStateChange {
 public:
  AWS_ECS_API ContainerStateChange() = default;
  AWS_ECS_API ContainerStateChange(Aws::Utils::Json::JsonView jsonValue);
  AWS_ECS_API ContainerStateChange& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_ECS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the container.</p>
   */
  inline const Aws::String& GetContainerName() const { return m_containerName; }
  inline bool ContainerNameHasBeenSet() const { return m_containerNameHasBeenSet; }
  template <typename ContainerNameT = Aws::String>
  void SetContainerName(ContainerNameT&& value) {
    m_containerNameHasBeenSet = true;
    m_containerName = std::forward<ContainerNameT>(value);
  }
  template <typename ContainerNameT = Aws::String>
  ContainerStateChange& WithContainerName(ContainerNameT&& value) {
    SetContainerName(std::forward<ContainerNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The container image SHA 256 digest.</p>
   */
  inline const Aws::String& GetImageDigest() const { return m_imageDigest; }
  inline bool ImageDigestHasBeenSet() const { return m_imageDigestHasBeenSet; }
  template <typename ImageDigestT = Aws::String>
  void SetImageDigest(ImageDigestT&& value) {
    m_imageDigestHasBeenSet = true;
    m_imageDigest = std::forward<ImageDigestT>(value);
  }
  template <typename ImageDigestT = Aws::String>
  ContainerStateChange& WithImageDigest(ImageDigestT&& value) {
    SetImageDigest(std::forward<ImageDigestT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Docker container.</p>
   */
  inline const Aws::String& GetRuntimeId() const { return m_runtimeId; }
  inline bool RuntimeIdHasBeenSet() const { return m_runtimeIdHasBeenSet; }
  template <typename RuntimeIdT = Aws::String>
  void SetRuntimeId(RuntimeIdT&& value) {
    m_runtimeIdHasBeenSet = true;
    m_runtimeId = std::forward<RuntimeIdT>(value);
  }
  template <typename RuntimeIdT = Aws::String>
  ContainerStateChange& WithRuntimeId(RuntimeIdT&& value) {
    SetRuntimeId(std::forward<RuntimeIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The exit code for the container, if the state change is a result of the
   * container exiting.</p>
   */
  inline int GetExitCode() const { return m_exitCode; }
  inline bool ExitCodeHasBeenSet() const { return m_exitCodeHasBeenSet; }
  inline void SetExitCode(int value) {
    m_exitCodeHasBeenSet = true;
    m_exitCode = value;
  }
  inline ContainerStateChange& WithExitCode(int value) {
    SetExitCode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Any network bindings that are associated with the container.</p>
   */
  inline const Aws::Vector<NetworkBinding>& GetNetworkBindings() const { return m_networkBindings; }
  inline bool NetworkBindingsHasBeenSet() const { return m_networkBindingsHasBeenSet; }
  template <typename NetworkBindingsT = Aws::Vector<NetworkBinding>>
  void SetNetworkBindings(NetworkBindingsT&& value) {
    m_networkBindingsHasBeenSet = true;
    m_networkBindings = std::forward<NetworkBindingsT>(value);
  }
  template <typename NetworkBindingsT = Aws::Vector<NetworkBinding>>
  ContainerStateChange& WithNetworkBindings(NetworkBindingsT&& value) {
    SetNetworkBindings(std::forward<NetworkBindingsT>(value));
    return *this;
  }
  template <typename NetworkBindingsT = NetworkBinding>
  ContainerStateChange& AddNetworkBindings(NetworkBindingsT&& value) {
    m_networkBindingsHasBeenSet = true;
    m_networkBindings.emplace_back(std::forward<NetworkBindingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The reason for the state change.</p>
   */
  inline const Aws::String& GetReason() const { return m_reason; }
  inline bool ReasonHasBeenSet() const { return m_reasonHasBeenSet; }
  template <typename ReasonT = Aws::String>
  void SetReason(ReasonT&& value) {
    m_reasonHasBeenSet = true;
    m_reason = std::forward<ReasonT>(value);
  }
  template <typename ReasonT = Aws::String>
  ContainerStateChange& WithReason(ReasonT&& value) {
    SetReason(std::forward<ReasonT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the container.</p>
   */
  inline const Aws::String& GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  template <typename StatusT = Aws::String>
  void SetStatus(StatusT&& value) {
    m_statusHasBeenSet = true;
    m_status = std::forward<StatusT>(value);
  }
  template <typename StatusT = Aws::String>
  ContainerStateChange& WithStatus(StatusT&& value) {
    SetStatus(std::forward<StatusT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_containerName;

  Aws::String m_imageDigest;

  Aws::String m_runtimeId;

  int m_exitCode{0};

  Aws::Vector<NetworkBinding> m_networkBindings;

  Aws::String m_reason;

  Aws::String m_status;
  bool m_containerNameHasBeenSet = false;
  bool m_imageDigestHasBeenSet = false;
  bool m_runtimeIdHasBeenSet = false;
  bool m_exitCodeHasBeenSet = false;
  bool m_networkBindingsHasBeenSet = false;
  bool m_reasonHasBeenSet = false;
  bool m_statusHasBeenSet = false;
};

}  // namespace Model
}  // namespace ECS
}  // namespace Aws
