﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ecs/ECS_EXPORTS.h>
#include <aws/ecs/model/TargetType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace ECS {
namespace Model {

/**
 * <p>An attribute is a name-value pair that's associated with an Amazon ECS
 * object. Use attributes to extend the Amazon ECS data model by adding custom
 * metadata to your resources. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html#attributes">Attributes</a>
 * in the <i>Amazon Elastic Container Service Developer Guide</i>.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/Attribute">AWS API
 * Reference</a></p>
 */
class Attribute {
 public:
  AWS_ECS_API Attribute() = default;
  AWS_ECS_API Attribute(Aws::Utils::Json::JsonView jsonValue);
  AWS_ECS_API Attribute& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_ECS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the attribute. The <code>name</code> must contain between 1 and
   * 128 characters. The name may contain letters (uppercase and lowercase), numbers,
   * hyphens (-), underscores (_), forward slashes (/), back slashes (\), or periods
   * (.).</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  Attribute& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The value of the attribute. The <code>value</code> must contain between 1 and
   * 128 characters. It can contain letters (uppercase and lowercase), numbers,
   * hyphens (-), underscores (_), periods (.), at signs (@), forward slashes (/),
   * back slashes (\), colons (:), or spaces. The value can't start or end with a
   * space.</p>
   */
  inline const Aws::String& GetValue() const { return m_value; }
  inline bool ValueHasBeenSet() const { return m_valueHasBeenSet; }
  template <typename ValueT = Aws::String>
  void SetValue(ValueT&& value) {
    m_valueHasBeenSet = true;
    m_value = std::forward<ValueT>(value);
  }
  template <typename ValueT = Aws::String>
  Attribute& WithValue(ValueT&& value) {
    SetValue(std::forward<ValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of the target to attach the attribute with. This parameter is
   * required if you use the short form ID for a resource instead of the full
   * ARN.</p>
   */
  inline TargetType GetTargetType() const { return m_targetType; }
  inline bool TargetTypeHasBeenSet() const { return m_targetTypeHasBeenSet; }
  inline void SetTargetType(TargetType value) {
    m_targetTypeHasBeenSet = true;
    m_targetType = value;
  }
  inline Attribute& WithTargetType(TargetType value) {
    SetTargetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the target. You can specify the short form ID for a resource or the
   * full Amazon Resource Name (ARN).</p>
   */
  inline const Aws::String& GetTargetId() const { return m_targetId; }
  inline bool TargetIdHasBeenSet() const { return m_targetIdHasBeenSet; }
  template <typename TargetIdT = Aws::String>
  void SetTargetId(TargetIdT&& value) {
    m_targetIdHasBeenSet = true;
    m_targetId = std::forward<TargetIdT>(value);
  }
  template <typename TargetIdT = Aws::String>
  Attribute& WithTargetId(TargetIdT&& value) {
    SetTargetId(std::forward<TargetIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_value;

  TargetType m_targetType{TargetType::NOT_SET};

  Aws::String m_targetId;
  bool m_nameHasBeenSet = false;
  bool m_valueHasBeenSet = false;
  bool m_targetTypeHasBeenSet = false;
  bool m_targetIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace ECS
}  // namespace Aws
